﻿let configManager;
let formValidator;

document.addEventListener('DOMContentLoaded', function () {
    configManager = new ConfigManager();
    formValidator = new FormValidationManager();
    formValidator.init();
});

// Form submission handler
document.getElementById('configForm').addEventListener('submit', async (e) => {
    if (formValidator.validateAllFields()) {
        e.preventDefault();
        await configManager.saveConfig();
    }
});
class ConfigManager {
    constructor() {
        this.loadConfig();
        this.initPasswordToggles();
        this.initHelpButton();
    }
    initHelpButton() {
        const helpButton = document.getElementById('helpButton');
        if (helpButton) {
            helpButton.addEventListener('click', () => {
                const docsUrl = 'https://buildin.ai/getit/share/d6e73283-8294-4bac-b1fd-98fca8c0a9e4';
                window.open(docsUrl, '_blank');
            });
        }
    }

    async loadConfig() {
        try {
            const response = await fetch('/api/config');
            const result = await response.json();
            const isSuccess = result.success || result.Success;
            const data = result.data || result.Data;

            if (isSuccess && data) {
                this.populateForm(data);
            }
        } catch (error) {
            console.error('Error loading config:', error);
        }
    }

    initPasswordToggles() {
        document.querySelectorAll('.password-toggle').forEach(toggle => {
            toggle.addEventListener('click', (e) => {
                e.preventDefault();
                const targetId = toggle.getAttribute('data-target');
                const input = document.getElementById(targetId);
                const container = toggle.closest('.password-container');

                if (input && container) {
                    if (input.type === 'password') {
                        input.type = 'text';
                        container.classList.add('showing');
                        toggle.innerHTML = '<span class="eye-icon">🔒</span>';
                    } else {
                        input.type = 'password';
                        container.classList.remove('showing');
                        toggle.innerHTML = '<span class="eye-icon">👁️</span>';
                    }
                }
            });
        });
    }

    populateForm(config) {
        console.log('Populating form with config:', config);

        // Функция для безопасной установки значения
        const setValue = (id, config) => {
            const element = document.getElementById(id);
            if (!element) {
                console.warn(`Element with id ${id} not found`);
                return;
            }

            // Находим значение в config независимо от регистра
            const configKey = Object.keys(config).find(key =>
                key.toLowerCase() === id.toLowerCase()
            );

            let value = configKey ? config[configKey] : null;

            if (element.type === 'checkbox') {
                element.checked = Boolean(value);
                console.log(`Set checkbox ${id} to ${element.checked} (from key: ${configKey})`);
            } else {
                element.value = value || '';
                console.log(`Set input ${id} to ${element.value} (from key: ${configKey})`);
            }
        };

        setValue('webSocketServerHost', config);
        setValue('webSocketServerPort', config);

        // iiko Settings
        setValue('iikoPin', config);
        setValue('stopListAmount', config);
        setValue('useAllProducts', config);
        setValue('packageGroupId', config);
        setValue('printOrderNumber', config);
        setValue('useKitchenSizeNames', config);
        setValue('iikoBizOrganizationId', config);

        // Обработка массива prohibitedSizes (особый случай)
        const prohibitedSizesElement = document.getElementById('prohibitedSizes');
        if (prohibitedSizesElement) {
            const prohibitedSizesKey = Object.keys(config).find(key =>
                key.toLowerCase() === 'prohibitedsizes'
            );
            if (prohibitedSizesKey && Array.isArray(config[prohibitedSizesKey])) {
                prohibitedSizesElement.value = config[prohibitedSizesKey].join(', ');
                console.log(`Set prohibitedSizes to ${prohibitedSizesElement.value}`);
            }
        }

        // Transport API Settings
        setValue('transportApiKey', config);
        setValue('transportOrganizationId', config);
        setValue('transportProgramId', config);
        setValue('transportBaseUrl', config);

        // iikoBiz Settings
        setValue('iikoBizBaseUrl', config);
        setValue('iikoBizUserId', config);
        setValue('iikoBizUserSecret', config);

        // API Notification Settings
        setValue('apiNotificationUrl', config);
        setValue('showAllModifiers', config);
        setValue('adultProductTags', config);
        setValue('applyStopList', config);
        setValue('reOpenShiftSignal', config);
        setValue('sendReturnPaymentNotification', config);

        // Barcode Settings
        setValue('barcodePrefixForWeightGoods', config);
        setValue('barcodePrefixForPieceGoods', config);
        setValue('includedToSendMeasuringUnitNames', config);

        // Order Settings
        setValue('generateRandomOrderNumber', config);
        setValue('closeOrderMode', config);
        setValue('preorderMode', config);

        // Выпадающие списки
        const loyaltyProviderElement = document.getElementById('loyaltyProvider');
        if (loyaltyProviderElement) {
            const loyaltyProviderKey = Object.keys(config).find(key =>
                key.toLowerCase() === 'loyaltyprovider'
            );
            loyaltyProviderElement.value = loyaltyProviderKey ? config[loyaltyProviderKey] : 'None';
            console.log(`Set loyaltyProvider to ${loyaltyProviderElement.value}`);
        }

        setValue('basketDelay', config);

        // Обработка TimeSpan полей
        const setTimeValue = (id, config) => {
            const element = document.getElementById(id);
            if (element) {
                const configKey = Object.keys(config).find(key =>
                    key.toLowerCase() === id.toLowerCase()
                );
                let timeValue = configKey ? config[configKey] : null;

                if (typeof timeValue === 'string') {
                    element.value = timeValue;
                } else if (timeValue && typeof timeValue === 'object') {
                    // Если TimeSpan представлен как объект
                    const hours = (timeValue.days * 24 + (timeValue.hours || 0)).toString().padStart(2, '0');
                    const minutes = (timeValue.minutes || 0).toString().padStart(2, '0');
                    const seconds = (timeValue.seconds || 0).toString().padStart(2, '0');
                    element.value = `${hours}:${minutes}:${seconds}`;
                } else {
                    element.value = timeValue || '';
                }
                console.log(`Set time field ${id} to ${element.value} (from key: ${configKey})`);
            }
        };

        setTimeValue('delayedPickupTime', config);
        setTimeValue('expectedPickupTime', config);
        setTimeValue('openCafeSessionTime', config);
        setTimeValue('closeCafeSessionTime', config);

        setValue('defaultDeliveryProductId', config);
        setValue('maxOrderNumberLength', config);
        setValue('removeComboTriggerGroup', config);
        setValue('useZeroValueForProductWithoutTax', config);

        // GetHub Settings
        setValue('getHubApiKey', config);
        setValue('getHubOrganizationId', config);
        setValue('getHubMenuUrl', config);
        setValue('getHubMenuName', config);
        setValue('getHubAdditionalMenuName', config);

        // System Settings
        setValue('syncTimeoutMinutes', config);
        setValue('registerPortUsersGroupName', config);
        setValue('portReservedMessage', config);
        setValue('defaultUserPhoneForCreateDelivery', config);
        setValue('trustPricesFromIiko', config);
        setValue('ignoreProductsTax', config);
        setValue('useExternalMarkingPlugin', config);
        setValue('useChequeTaskProcessorForAllOrders', config);
        setValue('defaultProductTax', config);

        // Выпадающий список для LocatorAddingType
        const locatorAddingTypeElement = document.getElementById('locatorAddingType');
        if (locatorAddingTypeElement) {
            const locatorAddingTypeKey = Object.keys(config).find(key =>
                key.toLowerCase() === 'locatoraddingtype'
            );
            locatorAddingTypeElement.value = locatorAddingTypeKey ? config[locatorAddingTypeKey] : 'ToExternalNumber';
            console.log(`Set locatorAddingType to ${locatorAddingTypeElement.value}`);
        }

        setValue('frontIsAgentMode', config);
        setValue('exitFromServiceModeWhenCafeSessionOpened', config);
        setValue('locatorText', config);

        console.log('Form population completed');
    }

    async saveConfig() {
        try {
            console.log('Saving configuration...');

            // Функция для безопасного получения значения
            const getValue = (id, type = 'string') => {
                const element = document.getElementById(id);
                if (!element) {
                    console.warn(`Element with id ${id} not found, returning default value`);
                    return null;
                }

                if (element.type === 'checkbox') {
                    return element.checked;
                }

                const value = element.value;
                if (type === 'number') {
                    return value ? parseInt(value) : 0;
                }
                if (type === 'float') {
                    return value ? parseFloat(value) : 0;
                }
                return value;
            };

            const config = {
                webSocketServerHost: getValue('webSocketServerHost'),
                webSocketServerPort: getValue('webSocketServerPort', 'number'),
                iikoPin: getValue('iikoPin'),
                stopListAmount: getValue('stopListAmount', 'number'),
                useAllProducts: getValue('useAllProducts'),
                transportApiKey: getValue('transportApiKey'),
                transportOrganizationId: getValue('transportOrganizationId'),
                transportProgramId: getValue('transportProgramId'),
                packageGroupId: getValue('packageGroupId'),
                printOrderNumber: getValue('printOrderNumber'),
                useKitchenSizeNames: getValue('useKitchenSizeNames'),
                iikoBizUserId: getValue('iikoBizUserId'),
                iikoBizUserSecret: getValue('iikoBizUserSecret'),
                iikoBizOrganizationId: getValue('iikoBizOrganizationId'),
                prohibitedSizes: getValue('prohibitedSizes') ?
                    getValue('prohibitedSizes').split(',').map(s => s.trim()).filter(s => s) : [],
                apiNotificationUrl: getValue('apiNotificationUrl'),
                showAllModifiers: getValue('showAllModifiers'),
                adultProductTags: getValue('adultProductTags'),
                applyStopList: getValue('applyStopList'),
                reOpenShiftSignal: getValue('reOpenShiftSignal'),
                sendReturnPaymentNotification: getValue('sendReturnPaymentNotification'),
                transportBaseUrl: getValue('transportBaseUrl'),
                iikoBizBaseUrl: getValue('iikoBizBaseUrl'),
                barcodePrefixForWeightGoods: getValue('barcodePrefixForWeightGoods', 'number'),
                barcodePrefixForPieceGoods: getValue('barcodePrefixForPieceGoods', 'number'),
                includedToSendMeasuringUnitNames: getValue('includedToSendMeasuringUnitNames'),
                generateRandomOrderNumber: getValue('generateRandomOrderNumber'),
                closeOrderMode: getValue('closeOrderMode'),
                preorderMode: getValue('preorderMode'),
                loyaltyProvider: getValue('loyaltyProvider'),
                getHubApiKey: getValue('getHubApiKey'),
                getHubOrganizationId: getValue('getHubOrganizationId'),
                getHubMenuUrl: getValue('getHubMenuUrl'),
                getHubMenuName: getValue('getHubMenuName'),
                getHubAdditionalMenuName: getValue('getHubAdditionalMenuName'),
                basketDelay: getValue('basketDelay', 'number'),
                delayedPickupTime: getValue('delayedPickupTime') || "01:00:00",
                defaultDeliveryProductId: getValue('defaultDeliveryProductId'),
                maxOrderNumberLength: getValue('maxOrderNumberLength', 'number'),
                expectedPickupTime: getValue('expectedPickupTime') || "00:00:00",
                removeComboTriggerGroup: getValue('removeComboTriggerGroup'),
                useZeroValueForProductWithoutTax: getValue('useZeroValueForProductWithoutTax'),
                syncTimeoutMinutes: getValue('syncTimeoutMinutes', 'number'),
                registerPortUsersGroupName: getValue('registerPortUsersGroupName'),
                portReservedMessage: getValue('portReservedMessage'),
                defaultUserPhoneForCreateDelivery: getValue('defaultUserPhoneForCreateDelivery'),
                trustPricesFromIiko: getValue('trustPricesFromIiko'),
                ignoreProductsTax: getValue('ignoreProductsTax'),
                useExternalMarkingPlugin: getValue('useExternalMarkingPlugin'),
                useChequeTaskProcessorForAllOrders: getValue('useChequeTaskProcessorForAllOrders'),
                defaultProductTax: getValue('defaultProductTax', 'float'),
                locatorAddingType: getValue('locatorAddingType'),
                openCafeSessionTime: getValue('openCafeSessionTime') || "00:00:00",
                closeCafeSessionTime: getValue('closeCafeSessionTime') || "00:00:00",
                frontIsAgentMode: getValue('frontIsAgentMode'),
                exitFromServiceModeWhenCafeSessionOpened: getValue('exitFromServiceModeWhenCafeSessionOpened'),
                locatorText: getValue('locatorText')
            };

            console.log('Config to save:', config);

            // Удаляем null значения
            Object.keys(config).forEach(key => {
                if (config[key] === null) {
                    delete config[key];
                }
            });

            const response = await fetch('/api/config', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(config)
            });

            if (!response.ok) {
                alert('Ошибка сохранения настроек');
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            const resultText = await response.text();
            console.log('Save response:', resultText);

            let result;
            try {
                result = JSON.parse(resultText);
            } catch (e) {
                // Если ответ не JSON, попробуем извлечь из HTML
                const jsonMatch = resultText.match(/\{[\s\S]*\}/);
                if (jsonMatch) {
                    result = JSON.parse(jsonMatch[0]);
                } else {
                    throw new Error('Invalid JSON response from server');
                }
            }
            alert('Настройки успешно обновлены');
            this.showResult(result.message, result.success);
        } catch (error) {
            console.error('Error saving config:', error);
            this.showResult('Error saving configuration: ' + error.message, false);
        }
    }

    showResult(message, isSuccess) {
        const resultElement = document.getElementById('saveResult');
        if (resultElement) {
            resultElement.textContent = message;
            resultElement.className = `result-message ${isSuccess ? 'success' : 'error'}`;
            resultElement.style.display = 'block';

            setTimeout(() => {
                resultElement.style.display = 'none';
            }, 5000);
        }

        // Также показываем в консоли
        console.log(`Result: ${message}, Success: ${isSuccess}`);
    }
}
class FormValidationManager {
    constructor() {
        this.uuidFields = [];
        this.uuidArrayFields = [];
        this.timeFields = [];
        this.errors = [];
    }

    init() {
        this.findAndInitFields();
        this.setupFieldValidation();
        this.setupFormValidation();
    }

    // Находим все поля для валидации
    findAndInitFields() {
        // Одиночные UUID поля
        console.log(`Одиночные UUID поля`);
        this.uuidFields = Array.from(document.querySelectorAll([
            'input[data-validate="uuid"]'
        ].join(',')));

        console.log(`Массивы UUID`);
        // Массивы UUID
        this.uuidArrayFields = Array.from(document.querySelectorAll([
            'input[data-validate="uuid-array"]'
        ].join(',')));

        console.log(`Поля времени`);
        // Поля времени
        this.timeFields = Array.from(document.querySelectorAll([
            'input[data-validate="time"]'
        ].join(',')));

        this.phoneFields = Array.from(document.querySelectorAll([
            'input[data-validate="phone"]',
        ].join(',')));

        console.log(`Found: ${this.uuidFields.length} UUID, ${this.uuidArrayFields.length} UUID arrays, ${this.timeFields.length} time fields, ${this.phoneFields.length} phone fields`);
    }

    // Настраиваем валидацию для каждого поля
    setupFieldValidation() {
        // Одиночные UUID
        this.uuidFields.forEach(field => {
            field.addEventListener('blur', () => this.validateUUIDField(field));
            field.addEventListener('input', () => this.clearFieldError(field));
        });

        // Массивы UUID
        this.uuidArrayFields.forEach(field => {
            field.addEventListener('blur', () => this.validateUUIDArrayField(field));
            field.addEventListener('input', () => {
                this.clearFieldError(field);
                this.formatUUIDArrayField(field);
            });
        });

        // Поля времени
        this.timeFields.forEach(field => {
            field.addEventListener('blur', () => this.validateTimeField(field));
            field.addEventListener('input', () => {
                this.clearFieldError(field);
                this.formatTimeField(field);
            });
        });

        this.phoneFields.forEach(field => {
            field.addEventListener('blur', () => this.validatePhoneField(field));
            field.addEventListener('input', (e) => {
                this.clearFieldError(field);
                //this.formatPhoneField(field, e);
            });
        });
    }
    validatePhoneField(field) {
        const value = field.value.trim();
        const isValid = FormValidator.isValidPhone(value);

        if (!isValid && value !== '') {
            this.markFieldAsInvalid(field,
                'Некорректный формат телефона. Пример: +7 (912) 345-67-89');
        } else {
            this.markFieldAsValid(field);
        }

        return isValid;
    }

    // Форматирование телефона в реальном времени
    formatPhoneField(field, event) {
        const input = event.target;
        const cursorPosition = input.selectionStart;
        const value = input.value;

        // Сохраняем позицию курсора
        const formatted = FormValidator.formatPhone(value);

        if (formatted !== value) {
            input.value = formatted;

            // Восстанавливаем позицию курсора с учетом добавленных символов
            const addedChars = formatted.length - value.length;
            input.setSelectionRange(cursorPosition + addedChars, cursorPosition + addedChars);
        }
    }

    // Ограничение ввода для телефона
    restrictPhoneInput(event) {
        const allowedChars = '0123456789+()- ';
        if (!allowedChars.includes(event.key)) {
            event.preventDefault();
        }
    }

    // Валидация массива UUID
    validateUUIDArrayField(field) {
        const value = field.value.trim();
        const isValid = FormValidator.isValidUUIDArray(value);

        if (!isValid) {
            const invalidUuids = this.getInvalidUUIDsFromArray(value);
            this.markFieldAsInvalid(field,
                `Некорректные UUID: ${invalidUuids.join(', ')}. Формат: UUID1, UUID2, UUID3`);
        } else {
            this.markFieldAsValid(field);
        }

        return isValid;
    }

    // Получить невалидные UUID из массива
    getInvalidUUIDsFromArray(value) {
        if (!value) return [];

        return value.split(',')
            .map(uuid => uuid.trim())
            .filter(uuid => uuid !== '')
            .filter(uuid => !FormValidator.isValidUUID(uuid));
    }

    // Форматирование массива UUID
    formatUUIDArrayField(field) {
        const formatted = FormValidator.formatUUIDArray(field.value);
        if (formatted && formatted !== field.value) {
            field.value = formatted;
        }
    }

    // Остальные методы остаются без изменений
    validateUUIDField(field) {
        const value = field.value.trim();
        const isValid = FormValidator.isValidUUID(value);

        if (!isValid) {
            this.markFieldAsInvalid(field, 'Некорректный формат UUID. Пример: 00000000-0000-0000-0000-000000000000');
        } else {
            this.markFieldAsValid(field);
        }

        return isValid;
    }

    validateTimeField(field) {
        const value = field.value.trim();
        const isValid = FormValidator.isValidTime(value);

        if (!isValid) {
            this.markFieldAsInvalid(field, 'Некорректный формат времени. Допустимый диапазон: 00:00:00 - 23:59:59');
        } else {
            this.markFieldAsValid(field);
        }

        return isValid;
    }

    formatTimeField(field) {
        const formatted = FormValidator.formatTime(field.value);
        if (formatted && formatted !== field.value) {
            field.value = formatted;
        }
    }

    
    markFieldAsInvalid(field, message) {
        field.classList.add('error');
        field.classList.remove('success');
        this.showFieldError(field, message);
    }

    markFieldAsValid(field) {
        field.classList.remove('error');
        field.classList.add('success');
        this.clearFieldError(field);
    }

    showFieldError(field, message) {
        this.clearFieldError(field);

        const errorElement = document.createElement('div');
        errorElement.className = 'field-error';
        errorElement.style.color = '#dc3545';
        errorElement.style.fontSize = '12px';
        errorElement.style.marginTop = '4px';
        errorElement.textContent = message;

        field.parentNode.appendChild(errorElement);
    }

    clearFieldError(field) {
        const errorElement = field.parentNode.querySelector('.field-error');
        if (errorElement) {
            errorElement.remove();
        }
    }

    // Валидация всех полей
    validateAllFields() {
        this.errors = [];

        this.uuidFields.forEach(field => {
            if (!this.validateUUIDField(field)) {
                this.errors.push({ field, message: 'Некорректный UUID', type: 'uuid' });
            }
        });

        this.uuidArrayFields.forEach(field => {
            if (!this.validateUUIDArrayField(field)) {
                this.errors.push({ field, message: 'Некорректный массив UUID', type: 'uuid-array' });
            }
        });

        this.timeFields.forEach(field => {
            if (!this.validateTimeField(field)) {
                this.errors.push({ field, message: 'Некорректное время', type: 'time' });
            }
        });

        this.phoneFields.forEach(field => {
            if (!this.validatePhoneField(field)) {
                this.errors.push({ field, message: 'Некорректный номер телефона', type: 'phone' });
            }
        });

        return this.errors.length === 0;
    }

    showAllErrors() {
        this.errors.forEach(error => {
            // Для массивов UUID показываем конкретные невалидные значения
            if (error.type === 'uuid-array') {
                const invalidUuids = this.getInvalidUUIDsFromArray(error.field.value);
                this.showFieldError(error.field,
                    `Некорректные UUID: ${invalidUuids.join(', ')}`);
            } else {
                this.showFieldError(error.field, error.message);
            }

            error.field.scrollIntoView({ behavior: 'smooth', block: 'center' });
        });

        if (this.errors.length > 0) {
            this.errors[0].field.focus();
        }
    }

    setupFormValidation() {
        const forms = document.querySelectorAll('form');

        forms.forEach(form => {
            form.addEventListener('submit', (e) => {
                if (!this.validateAllFields()) {
                    e.preventDefault();
                    this.showAllErrors();
                    alert(`Найдено ${this.errors.length} ошибок в форме. Пожалуйста, проверьте выделенные поля.`);
                }
            });
        });
    }
}

class PhoneUtils {
    // Нормализация номера (только цифры)
    static normalizePhone(phone) {
        if (!phone) return '';
        return phone.replace(/[^\d]/g, '');
    }

    // Проверка на мобильный номер (Россия)
    static isMobileRussian(phone) {
        const normalized = this.normalizePhone(phone);
        const mobilePrefixes = ['9', '79', '89', '99'];
        return mobilePrefixes.some(prefix => normalized.startsWith(prefix));
    }

    // Получить код оператора (Россия)
    static getOperatorCode(phone) {
        const normalized = this.normalizePhone(phone);
        if (normalized.startsWith('7') || normalized.startsWith('8')) {
            return normalized.substring(1, 4);
        }
        return null;
    }

    // Маска для ввода в реальном времени
    static createPhoneMask(value) {
        const numbers = value.replace(/\D/g, '');
        let formatted = '';

        if (numbers.length > 0) {
            formatted = '+7 ';
        }
        if (numbers.length > 1) {
            formatted += '(' + numbers.substring(1, 4);
        }
        if (numbers.length >= 4) {
            formatted += ') ';
        }
        if (numbers.length >= 7) {
            formatted += numbers.substring(4, 7) + '-';
        }
        if (numbers.length >= 9) {
            formatted += numbers.substring(7, 9) + '-';
        }
        if (numbers.length >= 11) {
            formatted += numbers.substring(9, 11);
        }

        return formatted;
    }
}
class FormValidator {
    static uuidRegex = /^[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i;

    // Валидация одиночного UUID
    static isValidUUID(value) {
        if (value === null || value === undefined || value === '') {
            return true;
        }
        if (typeof value !== 'string') return false;
        return this.uuidRegex.test(value.trim());
    }

    // Валидация массива UUID через запятую
    static isValidUUIDArray(value) {
        if (value === null || value === undefined || value === '') {
            return true; // Пустое значение - валидно
        }
        if (typeof value !== 'string') return false;

        const uuids = value.split(',').map(uuid => uuid.trim()).filter(uuid => uuid !== '');

        // Если после split остались только пустые строки - считаем валидным
        if (uuids.length === 0) return true;

        // Проверяем каждый UUID
        return uuids.every(uuid => this.uuidRegex.test(uuid));
    }

    // Форматирование массива UUID (убирает пробелы, сортирует)
    static formatUUIDArray(value) {
        if (!value) return '';

        const uuids = value.split(',')
            .map(uuid => uuid.trim())
            .filter(uuid => uuid !== '')
            .sort();

        return uuids.join(', ');
    }

    // Валидация времени
    static isValidTime(value) {
        if (!value || value.trim() === '') return true;

        const timeRegex = /^([01]?[0-9]|2[0-3]):([0-5][0-9]):([0-5][0-9])$/;
        if (!timeRegex.test(value)) return false;

        const [hours, minutes, seconds] = value.split(':').map(Number);
        return hours >= 0 && hours <= 23 &&
            minutes >= 0 && minutes <= 59 &&
            seconds >= 0 && seconds <= 59;
    }

    static formatTime(value) {
        if (!value) return '';

        const cleanValue = value.replace(/[^0-9:]/g, '');
        const parts = cleanValue.split(':').filter(part => part !== '');

        if (parts.length === 0) return '';

        const hours = parts[0] ? parts[0].padStart(2, '0').substr(0, 2) : '00';
        const minutes = parts[1] ? parts[1].padStart(2, '0').substr(0, 2) : '00';
        const seconds = parts[2] ? parts[2].padStart(2, '0').substr(0, 2) : '00';

        return `${hours}:${minutes}:${seconds}`;
    }

    static isValidPhone(value) {
        if (value === null || value === undefined || value === '') {
            return true; // Пустое значение - валидно
        }
        if (typeof value !== 'string') return false;

        // Очищаем номер от всего кроме цифр и +
        const cleanPhone = value.replace(/[^\d+]/g, '');

        // Проверяем различные форматы
        const phoneRegex = /^\+?[0-9]{10,15}$/;
        return phoneRegex.test(value);
    }

    // Форматирование телефона в российский стандарт
    static formatPhone(value) {
        if (!value) return '';

        // Оставляем только цифры и +
        const cleanPhone = value.replace(/[^\d+]/g, '');

        // Если номер начинается с 8, меняем на +7
        let formatted = cleanPhone;
        if (formatted.startsWith('8') && formatted.length === 11) {
            formatted = '+7' + formatted.substring(1);
        }

        // Если номер без кода страны, добавляем +7
        if (/^\d{10}$/.test(formatted)) {
            formatted = '+7' + formatted;
        }

        // Форматируем в +7 (XXX) XXX-XX-XX
        if (formatted.startsWith('+7') && formatted.length === 12) {
            const match = formatted.match(/^\+7(\d{3})(\d{3})(\d{2})(\d{2})$/);
            if (match) {
                return `+7 (${match[1]}) ${match[2]}-${match[3]}-${match[4]}`;
            }
        }

        return formatted;
    }

    // Упрощенная валидация (только российские номера)
    static isValidRussianPhone(value) {
        if (value === null || value === undefined || value === '') {
            return true;
        }
        if (typeof value !== 'string') return false;

        const cleanPhone = value.replace(/[^\d+]/g, '');

        // Российские форматы: +7..., 8..., 7...
        const russianRegex = /^(\+7|8|7)\d{10}$/;
        return russianRegex.test(cleanPhone);
    }

}
